/*
 * SPDX-FileCopyrightText: 2024 Espressif Systems (Shanghai) CO LTD
 *
 * SPDX-License-Identifier: Apache-2.0
 */

#pragma once
#ifdef __cplusplus
extern "C" {
#endif

#include "esp_zigbee_type.h"

/** @brief Price Cluster client attributes */
typedef enum {
    ESP_ZB_ZCL_ATTR_PRICE_CLI_PRICE_INCREASE_RANDOMIZE_MINUTES  = 0x0000,   /**< This attribute represents the maximum amount of time to be used
                                                                             *  when randomizing the response to a price increase.
                                                                             */
    ESP_ZB_ZCL_ATTR_PRICE_CLI_PRICE_DECREASE_RANDOMIZE_MINUTES  = 0x0001,   /**< This attribute represents the maximum number of minutes to be used
                                                                             *   when randomizing the response to a price decrease.
                                                                             */
    ESP_ZB_ZCL_ATTR_PRICE_CLI_COMMODITY_TYPE                    = 0x0002,   /**< This attribute provides a label for identifying the type of pricing
                                                                             *   client present.
                                                                             */
} esp_zb_zcl_price_cli_attr_t;


/** @brief Price Cluster General server attribute sets */
typedef enum {
    ESP_ZB_ZCL_PRICE_SET_TIER_LABEL                          = 0x00,    /**< Provides a method for utilities to assign a label to the Price Tier. */
    ESP_ZB_ZCL_PRICE_SET_BLOCK_THRESHOLD                     = 0x01,    /**< Provides remote access to the Price server Block Thresholds. */
    ESP_ZB_ZCL_PRICE_SET_BLOCK_PERIOD                        = 0x02,    /**< Provides remote access to the Price server Block Threshold period. */
    ESP_ZB_ZCL_PRICE_SET_COMMODITY                           = 0x03,    /**< Represents items that are associated with a particular commodity. */
    ESP_ZB_ZCL_PRICE_SET_BLOCK_PRICE_INFORMATION             = 0x04,    /**< Provides remote access to the block prices. */
    ESP_ZB_ZCL_PRICE_SET_EXTENDED_PRICE_INFORMATION          = 0x05,    /**< Represents extended price information. */
    ESP_ZB_ZCL_PRICE_SET_TARIFF_INFORMATION                  = 0x06,    /**< Represents items that are associated with a particular Price Tariff. */
    ESP_ZB_ZCL_PRICE_SET_BILLING_INFORMATION                 = 0x07,    /**< Provides remote access to the Price server Billing information. */
    ESP_ZB_ZCL_PRICE_SET_CREDIT_PAYMENT                      = 0x08,    /**< Provides a method for the HAN (IHD) to understand the current status
                                                                         *   of the credit-only payment made to the energy supplier. */
    ESP_ZB_ZCL_PRICE_SET_RECEIVED_TIER_LABEL                 = 0x80,    /**< Provides a method for utilities to assign a label to Received Price Tiers. */
    ESP_ZB_ZCL_PRICE_SET_RECEIVED_BLOCK_THRESHOLD            = 0x81,    /**< Provides remote access to the Price server ReceivedBlockThresholds. */
    ESP_ZB_ZCL_PRICE_SET_RECEIVED_BLOCK_PERIOD               = 0x82,    /**< Provides remote access to the Price server Received Block Threshold period. */
    ESP_ZB_ZCL_PRICE_SET_RECEIVED_BLOCK_PRICE_INFORMATION    = 0x84,    /**< See ESP_ZB_ZCL_PRICE_SET_BLOCK_PRICE_INFORMATION */
    ESP_ZB_ZCL_PRICE_SET_RECEIVED_EXTENDED_PRICE_INFORMATION = 0x85,    /**< See ESP_ZB_ZCL_PRICE_SET_EXTENDED_PRICE_INFORMATION */
    ESP_ZB_ZCL_PRICE_SET_RECEIVED_TARIFF_INFORMATION         = 0x86,    /**< See ESP_ZB_ZCL_PRICE_SET_TARIFF_INFORMATION */
    ESP_ZB_ZCL_PRICE_SET_RECEIVED_BILLING_INFORMATION        = 0x87,    /**< See ESP_ZB_ZCL_PRICE_SET_BILLING_INFORMATION */
} esp_zb_zcl_price_srv_attr_set_t;


/** @brief Tier Label (Delivered) Attributes Set */
typedef enum {
    /** The TierNPriceLabel attributes provide a method for utilities to assign a
     *  label to the Price Tier declared within the Publish Price command.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER1_PRICE_LABEL = ESP_ZB_ZCL_ATTR_SET_WITH_ATTR_ID(ESP_ZB_ZCL_PRICE_SET_TIER_LABEL, 0x00),
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER2_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER3_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER4_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER5_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER6_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER7_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER8_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER9_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER10_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER11_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER12_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER13_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER14_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER15_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER16_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER17_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER18_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER19_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER20_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER21_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER22_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER23_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER24_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER25_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER26_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER27_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER28_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER29_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER30_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER31_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER32_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER33_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER34_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER35_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER36_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER37_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER38_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER39_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER40_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER41_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER42_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER43_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER44_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER45_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER46_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER47_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER48_PRICE_LABEL,
} esp_zb_zcl_price_set_tier_label_t;


/** @brief Block Threshold (Delivered) Attributes Set */
typedef enum {
    /** Attributes Block1Threshold through Block15Threshold represent the block
     * threshold values for a given period (typically the billing cycle).
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_BLOCK1_THRESHOLD = ESP_ZB_ZCL_ATTR_SET_WITH_ATTR_ID(ESP_ZB_ZCL_PRICE_SET_BLOCK_THRESHOLD, 0x00),
    ESP_ZB_ZCL_ATTR_PRICE_SRV_BLOCK2_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_BLOCK3_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_BLOCK4_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_BLOCK5_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_BLOCK6_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_BLOCK7_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_BLOCK8_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_BLOCK9_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_BLOCK10_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_BLOCK11_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_BLOCK12_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_BLOCK13_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_BLOCK14_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_BLOCK15_THRESHOLD,

    /** Where a single set of thresholds is used, the BlockThresholdCount
     *  attribute indicates the number of applicable BlockNThresholds. Where more
     *  than one set of thresholds is used, each set will be accompanied by an
     *  appropriate TierNBlockThresholdCount attribute
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_BLOCK_THRESHOLD_COUNT,

    /** Attributes Tier1Block1Threshold through Tier15Block15Threshold represent
     *  the block threshold values applicable to a specific TOU tier for a given
     *  period (typically the billing cycle).
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER1_BLOCK1_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER1_BLOCK2_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER1_BLOCK3_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER1_BLOCK4_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER1_BLOCK5_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER1_BLOCK6_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER1_BLOCK7_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER1_BLOCK8_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER1_BLOCK9_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER1_BLOCK10_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER1_BLOCK11_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER1_BLOCK12_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER1_BLOCK13_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER1_BLOCK14_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER1_BLOCK15_THRESHOLD,

    /** The TierNBlockThresholdCount attributes hold the number of block
     *  thresholds applicable to a given tier.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER1_BLOCK_THRESHOLD_COUNT,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER2_BLOCK1_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER2_BLOCK2_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER2_BLOCK3_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER2_BLOCK4_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER2_BLOCK5_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER2_BLOCK6_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER2_BLOCK7_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER2_BLOCK8_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER2_BLOCK9_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER2_BLOCK10_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER2_BLOCK11_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER2_BLOCK12_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER2_BLOCK13_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER2_BLOCK14_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER2_BLOCK15_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER2_BLOCK_THRESHOLD_COUNT,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER3_BLOCK1_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER3_BLOCK2_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER3_BLOCK3_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER3_BLOCK4_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER3_BLOCK5_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER3_BLOCK6_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER3_BLOCK7_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER3_BLOCK8_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER3_BLOCK9_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER3_BLOCK10_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER3_BLOCK11_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER3_BLOCK12_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER3_BLOCK13_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER3_BLOCK14_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER3_BLOCK15_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER3_BLOCK_THRESHOLD_COUNT,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER4_BLOCK1_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER4_BLOCK2_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER4_BLOCK3_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER4_BLOCK4_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER4_BLOCK5_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER4_BLOCK6_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER4_BLOCK7_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER4_BLOCK8_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER4_BLOCK9_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER4_BLOCK10_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER4_BLOCK11_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER4_BLOCK12_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER4_BLOCK13_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER4_BLOCK14_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER4_BLOCK15_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER4_BLOCK_THRESHOLD_COUNT,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER5_BLOCK1_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER5_BLOCK2_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER5_BLOCK3_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER5_BLOCK4_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER5_BLOCK5_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER5_BLOCK6_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER5_BLOCK7_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER5_BLOCK8_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER5_BLOCK9_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER5_BLOCK10_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER5_BLOCK11_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER5_BLOCK12_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER5_BLOCK13_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER5_BLOCK14_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER5_BLOCK15_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER5_BLOCK_THRESHOLD_COUNT,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER6_BLOCK1_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER6_BLOCK2_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER6_BLOCK3_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER6_BLOCK4_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER6_BLOCK5_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER6_BLOCK6_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER6_BLOCK7_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER6_BLOCK8_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER6_BLOCK9_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER6_BLOCK10_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER6_BLOCK11_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER6_BLOCK12_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER6_BLOCK13_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER6_BLOCK14_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER6_BLOCK15_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER6_BLOCK_THRESHOLD_COUNT,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER7_BLOCK1_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER7_BLOCK2_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER7_BLOCK3_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER7_BLOCK4_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER7_BLOCK5_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER7_BLOCK6_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER7_BLOCK7_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER7_BLOCK8_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER7_BLOCK9_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER7_BLOCK10_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER7_BLOCK11_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER7_BLOCK12_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER7_BLOCK13_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER7_BLOCK14_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER7_BLOCK15_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER7_BLOCK_THRESHOLD_COUNT,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER8_BLOCK1_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER8_BLOCK2_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER8_BLOCK3_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER8_BLOCK4_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER8_BLOCK5_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER8_BLOCK6_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER8_BLOCK7_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER8_BLOCK8_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER8_BLOCK9_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER8_BLOCK10_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER8_BLOCK11_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER8_BLOCK12_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER8_BLOCK13_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER8_BLOCK14_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER8_BLOCK15_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER8_BLOCK_THRESHOLD_COUNT,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER9_BLOCK1_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER9_BLOCK2_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER9_BLOCK3_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER9_BLOCK4_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER9_BLOCK5_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER9_BLOCK6_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER9_BLOCK7_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER9_BLOCK8_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER9_BLOCK9_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER9_BLOCK10_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER9_BLOCK11_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER9_BLOCK12_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER9_BLOCK13_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER9_BLOCK14_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER9_BLOCK15_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER9_BLOCK_THRESHOLD_COUNT,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER10_BLOCK1_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER10_BLOCK2_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER10_BLOCK3_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER10_BLOCK4_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER10_BLOCK5_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER10_BLOCK6_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER10_BLOCK7_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER10_BLOCK8_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER10_BLOCK9_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER10_BLOCK10_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER10_BLOCK11_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER10_BLOCK12_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER10_BLOCK13_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER10_BLOCK14_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER10_BLOCK15_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER10_BLOCK_THRESHOLD_COUNT,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER11_BLOCK1_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER11_BLOCK2_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER11_BLOCK3_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER11_BLOCK4_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER11_BLOCK5_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER11_BLOCK6_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER11_BLOCK7_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER11_BLOCK8_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER11_BLOCK9_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER11_BLOCK10_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER11_BLOCK11_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER11_BLOCK12_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER11_BLOCK13_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER11_BLOCK14_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER11_BLOCK15_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER11_BLOCK_THRESHOLD_COUNT,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER12_BLOCK1_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER12_BLOCK2_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER12_BLOCK3_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER12_BLOCK4_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER12_BLOCK5_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER12_BLOCK6_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER12_BLOCK7_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER12_BLOCK8_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER12_BLOCK9_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER12_BLOCK10_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER12_BLOCK11_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER12_BLOCK12_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER12_BLOCK13_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER12_BLOCK14_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER12_BLOCK15_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER12_BLOCK_THRESHOLD_COUNT,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER13_BLOCK1_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER13_BLOCK2_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER13_BLOCK3_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER13_BLOCK4_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER13_BLOCK5_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER13_BLOCK6_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER13_BLOCK7_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER13_BLOCK8_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER13_BLOCK9_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER13_BLOCK10_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER13_BLOCK11_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER13_BLOCK12_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER13_BLOCK13_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER13_BLOCK14_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER13_BLOCK15_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER13_BLOCK_THRESHOLD_COUNT,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER14_BLOCK1_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER14_BLOCK2_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER14_BLOCK3_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER14_BLOCK4_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER14_BLOCK5_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER14_BLOCK6_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER14_BLOCK7_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER14_BLOCK8_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER14_BLOCK9_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER14_BLOCK10_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER14_BLOCK11_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER14_BLOCK12_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER14_BLOCK13_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER14_BLOCK14_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER14_BLOCK15_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER14_BLOCK_THRESHOLD_COUNT,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER15_BLOCK1_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER15_BLOCK2_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER15_BLOCK3_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER15_BLOCK4_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER15_BLOCK5_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER15_BLOCK6_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER15_BLOCK7_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER15_BLOCK8_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER15_BLOCK9_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER15_BLOCK10_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER15_BLOCK11_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER15_BLOCK12_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER15_BLOCK13_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER15_BLOCK14_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER15_BLOCK15_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER15_BLOCK_THRESHOLD_COUNT,
} esp_zb_zcl_set_block_threshold_t;


/** @brief Block Period (Delivered) Attributes Set */
typedef enum {
    /** The StartOfBlockPeriod attribute represents the start time of the current
     *  block tariff period.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_START_OF_BLOCK_PERIOD = ESP_ZB_ZCL_ATTR_SET_WITH_ATTR_ID(ESP_ZB_ZCL_PRICE_SET_BLOCK_PERIOD, 0x00),
    ESP_ZB_ZCL_ATTR_PRICE_SRV_BLOCK_PERIOD_DURATION,        /** The BlockPeriodDuration attribute represents the current block tariff
                                                             *  period duration in units defined by the BlockPeriodDurationType attribute.
                                                             */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_THRESHOLD_MULTIPLIER,         /** ThresholdMultiplier provides a value to be multiplied against Threshold attributes.
                                                             */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_THRESHOLD_DIVISOR,            /** ThresholdDivisor provides a value to divide the result of applying
                                                             *  the ThresholdMultiplier attribute to Block Threshold values to derive
                                                             *  values That can be compared against the CurrentBlockPeriodConsumptionDelivered
                                                             *  attribute within the Metering cluster.
                                                             */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_BLOCK_PERIOD_DURATION_TYPE,   /** The BlockPeriodDurationType attribute indicates the timebase used for the
                                                             *  BlockPeriodDuration attribute.
                                                             */
} esp_zb_zcl_price_set_block_period_t;

/** @brief Values for BlockPeriodDurationType attribute
 *  @see ESP_ZB_ZCL_ATTR_PRICE_SRV_BLOCK_PERIOD_DURATION_TYPE
 */
typedef enum {
    ESP_ZB_ZCL_PRICE_BLOCK_PERIOD_DURATION_TYPE_MINUTES = 0x00,   /**< minutes (default) */
    ESP_ZB_ZCL_PRICE_BLOCK_PERIOD_DURATION_TYPE_DAYS    = 0x01,   /**< days */
    ESP_ZB_ZCL_PRICE_BLOCK_PERIOD_DURATION_TYPE_WEEKS   = 0x02,   /**< weeks */
    ESP_ZB_ZCL_PRICE_BLOCK_PERIOD_DURATION_TYPE_MONTHS  = 0x03,   /**< months */
} esp_zb_zcl_price_block_period_duration_type_t;


/** @brief Commodity Attributes Set */
typedef enum {
    /** CommodityType provides a label for identifying the type of pricing server
     *  present.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_COMMODITY_TYPE = ESP_ZB_ZCL_ATTR_SET_WITH_ATTR_ID(ESP_ZB_ZCL_PRICE_SET_COMMODITY, 0x00),

    /** The value of the Standing Charge is a daily fixed charge associated with
     *  supplying the commodity, measured in base unit of Currency with the decimal
     *  point located as indicated by the Trailing Digits field of a Publish Price
     *  command or PriceTrailingDigit attribute.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_STANDING_CHARGE,

    /** The conversion factor is used for gas meter and takes into account changes
     *  in the volume of gas based on temperature and pressure.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_CONVERSION_FACTOR,

    /** An 8-bit BitMap used to determine where the decimal point is located in
     *  the ConversionFactor attribute.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_CONVERSION_FACTOR_TRAILING_DIGIT,

    /** The amount of heat generated when a given mass of fuel is completely burned.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_CALORIFIC_VALUE,

    /** This attribute defines the unit for the CalorificValue.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_CALORIFIC_VALUE_UNIT,

    /** An 8-bit BitMap used to determine where the decimal point is located in
     *  the CalorificValue attribute.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_CALORIFIC_VALUE_TRAILING_DIGIT,
} esp_zb_zcl_price_set_commodity_t;

/** @brief Values for CalorificValueUnit attribute
 *  @see ESP_ZB_ZCL_ATTR_PRICE_SRV_CALORIFIC_VALUE_UNIT
 */
typedef enum {
    ESP_ZB_ZCL_PRICE_CALORIFIC_VALUE_UNIT_MJ_M3 = 0x01,  /**< MJ/m3 */
    ESP_ZB_ZCL_PRICE_CALORIFIC_VALUE_UNIT_MJ_KG,         /**< MJ/kg */
} esp_zb_zcl_price_calorific_value_unit_values_t;


/** @brief Block Price Information (Delivered) Attributes Set */
typedef enum {
    /** Attributes PriceNoTierBlock1 through PriceTier15Block16 represent the
     *  price of Energy, Gas, or Water delivered to the premises (i.e. delivered
     *  to the customer from the utility) at a specific price tier as defined by
     *  a TOU schedule, Block Threshold or a real time pricing period.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_NO_TIER_BLOCK1_PRICE = ESP_ZB_ZCL_ATTR_SET_WITH_ATTR_ID(ESP_ZB_ZCL_PRICE_SET_BLOCK_PRICE_INFORMATION, 0x00),
    ESP_ZB_ZCL_ATTR_PRICE_SRV_NO_TIER_BLOCK2_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_NO_TIER_BLOCK3_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_NO_TIER_BLOCK4_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_NO_TIER_BLOCK5_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_NO_TIER_BLOCK6_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_NO_TIER_BLOCK7_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_NO_TIER_BLOCK8_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_NO_TIER_BLOCK9_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_NO_TIER_BLOCK10_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_NO_TIER_BLOCK11_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_NO_TIER_BLOCK12_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_NO_TIER_BLOCK13_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_NO_TIER_BLOCK14_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_NO_TIER_BLOCK15_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_NO_TIER_BLOCK16_PRICE,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER1_BLOCK1_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER1_BLOCK2_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER1_BLOCK3_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER1_BLOCK4_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER1_BLOCK5_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER1_BLOCK6_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER1_BLOCK7_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER1_BLOCK8_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER1_BLOCK9_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER1_BLOCK10_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER1_BLOCK11_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER1_BLOCK12_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER1_BLOCK13_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER1_BLOCK14_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER1_BLOCK15_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER1_BLOCK16_PRICE,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER2_BLOCK1_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER2_BLOCK2_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER2_BLOCK3_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER2_BLOCK4_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER2_BLOCK5_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER2_BLOCK6_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER2_BLOCK7_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER2_BLOCK8_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER2_BLOCK9_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER2_BLOCK10_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER2_BLOCK11_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER2_BLOCK12_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER2_BLOCK13_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER2_BLOCK14_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER2_BLOCK15_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER2_BLOCK16_PRICE,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER3_BLOCK1_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER3_BLOCK2_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER3_BLOCK3_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER3_BLOCK4_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER3_BLOCK5_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER3_BLOCK6_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER3_BLOCK7_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER3_BLOCK8_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER3_BLOCK9_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER3_BLOCK10_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER3_BLOCK11_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER3_BLOCK12_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER3_BLOCK13_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER3_BLOCK14_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER3_BLOCK15_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER3_BLOCK16_PRICE,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER4_BLOCK1_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER4_BLOCK2_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER4_BLOCK3_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER4_BLOCK4_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER4_BLOCK5_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER4_BLOCK6_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER4_BLOCK7_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER4_BLOCK8_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER4_BLOCK9_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER4_BLOCK10_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER4_BLOCK11_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER4_BLOCK12_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER4_BLOCK13_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER4_BLOCK14_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER4_BLOCK15_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER4_BLOCK16_PRICE,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER5_BLOCK1_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER5_BLOCK2_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER5_BLOCK3_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER5_BLOCK4_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER5_BLOCK5_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER5_BLOCK6_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER5_BLOCK7_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER5_BLOCK8_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER5_BLOCK9_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER5_BLOCK10_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER5_BLOCK11_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER5_BLOCK12_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER5_BLOCK13_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER5_BLOCK14_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER5_BLOCK15_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER5_BLOCK16_PRICE,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER6_BLOCK1_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER6_BLOCK2_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER6_BLOCK3_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER6_BLOCK4_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER6_BLOCK5_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER6_BLOCK6_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER6_BLOCK7_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER6_BLOCK8_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER6_BLOCK9_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER6_BLOCK10_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER6_BLOCK11_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER6_BLOCK12_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER6_BLOCK13_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER6_BLOCK14_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER6_BLOCK15_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER6_BLOCK16_PRICE,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER7_BLOCK1_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER7_BLOCK2_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER7_BLOCK3_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER7_BLOCK4_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER7_BLOCK5_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER7_BLOCK6_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER7_BLOCK7_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER7_BLOCK8_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER7_BLOCK9_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER7_BLOCK10_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER7_BLOCK11_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER7_BLOCK12_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER7_BLOCK13_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER7_BLOCK14_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER7_BLOCK15_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER7_BLOCK16_PRICE,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER8_BLOCK1_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER8_BLOCK2_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER8_BLOCK3_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER8_BLOCK4_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER8_BLOCK5_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER8_BLOCK6_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER8_BLOCK7_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER8_BLOCK8_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER8_BLOCK9_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER8_BLOCK10_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER8_BLOCK11_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER8_BLOCK12_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER8_BLOCK13_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER8_BLOCK14_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER8_BLOCK15_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER8_BLOCK16_PRICE,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER9_BLOCK1_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER9_BLOCK2_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER9_BLOCK3_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER9_BLOCK4_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER9_BLOCK5_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER9_BLOCK6_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER9_BLOCK7_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER9_BLOCK8_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER9_BLOCK9_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER9_BLOCK10_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER9_BLOCK11_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER9_BLOCK12_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER9_BLOCK13_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER9_BLOCK14_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER9_BLOCK15_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER9_BLOCK16_PRICE,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER10_BLOCK1_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER10_BLOCK2_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER10_BLOCK3_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER10_BLOCK4_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER10_BLOCK5_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER10_BLOCK6_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER10_BLOCK7_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER10_BLOCK8_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER10_BLOCK9_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER10_BLOCK10_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER10_BLOCK11_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER10_BLOCK12_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER10_BLOCK13_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER10_BLOCK14_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER10_BLOCK15_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER10_BLOCK16_PRICE,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER11_BLOCK1_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER11_BLOCK2_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER11_BLOCK3_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER11_BLOCK4_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER11_BLOCK5_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER11_BLOCK6_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER11_BLOCK7_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER11_BLOCK8_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER11_BLOCK9_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER11_BLOCK10_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER11_BLOCK11_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER11_BLOCK12_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER11_BLOCK13_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER11_BLOCK14_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER11_BLOCK15_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER11_BLOCK16_PRICE,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER12_BLOCK1_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER12_BLOCK2_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER12_BLOCK3_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER12_BLOCK4_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER12_BLOCK5_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER12_BLOCK6_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER12_BLOCK7_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER12_BLOCK8_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER12_BLOCK9_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER12_BLOCK10_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER12_BLOCK11_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER12_BLOCK12_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER12_BLOCK13_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER12_BLOCK14_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER12_BLOCK15_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER12_BLOCK16_PRICE,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER13_BLOCK1_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER13_BLOCK2_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER13_BLOCK3_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER13_BLOCK4_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER13_BLOCK5_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER13_BLOCK6_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER13_BLOCK7_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER13_BLOCK8_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER13_BLOCK9_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER13_BLOCK10_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER13_BLOCK11_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER13_BLOCK12_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER13_BLOCK13_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER13_BLOCK14_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER13_BLOCK15_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER13_BLOCK16_PRICE,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER14_BLOCK1_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER14_BLOCK2_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER14_BLOCK3_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER14_BLOCK4_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER14_BLOCK5_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER14_BLOCK6_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER14_BLOCK7_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER14_BLOCK8_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER14_BLOCK9_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER14_BLOCK10_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER14_BLOCK11_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER14_BLOCK12_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER14_BLOCK13_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER14_BLOCK14_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER14_BLOCK15_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER14_BLOCK16_PRICE,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER15_BLOCK1_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER15_BLOCK2_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER15_BLOCK3_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER15_BLOCK4_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER15_BLOCK5_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER15_BLOCK6_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER15_BLOCK7_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER15_BLOCK8_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER15_BLOCK9_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER15_BLOCK10_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER15_BLOCK11_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER15_BLOCK12_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER15_BLOCK13_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER15_BLOCK14_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER15_BLOCK15_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER15_BLOCK16_PRICE,
} esp_zb_zcl_price_set_block_price_information_t;


/** @brief Extended Price Information (Delivered) Attributes Set */
typedef enum {
    /* reserved 0x0500 - 0x050E */

    /** Attributes PriceTier16 through PriceTier48 represent the price of Energy,
     *  Gas, or Water delivered to the premises (i.e. delivered to the customer
     *  from the utility) at a specific price tier.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_PRICE_TIER16 = ESP_ZB_ZCL_ATTR_SET_WITH_ATTR_ID(ESP_ZB_ZCL_PRICE_SET_EXTENDED_PRICE_INFORMATION, 0x0F),
    ESP_ZB_ZCL_ATTR_PRICE_SRV_PRICE_TIER17,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_PRICE_TIER18,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_PRICE_TIER19,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_PRICE_TIER20,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_PRICE_TIER21,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_PRICE_TIER22,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_PRICE_TIER23,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_PRICE_TIER24,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_PRICE_TIER25,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_PRICE_TIER26,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_PRICE_TIER27,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_PRICE_TIER28,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_PRICE_TIER29,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_PRICE_TIER30,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_PRICE_TIER31,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_PRICE_TIER32,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_PRICE_TIER33,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_PRICE_TIER34,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_PRICE_TIER35,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_PRICE_TIER36,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_PRICE_TIER37,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_PRICE_TIER38,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_PRICE_TIER39,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_PRICE_TIER40,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_PRICE_TIER41,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_PRICE_TIER42,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_PRICE_TIER43,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_PRICE_TIER44,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_PRICE_TIER45,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_PRICE_TIER46,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_PRICE_TIER47,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_PRICE_TIER48,
    /* reserved 0x0530 - 0x05FD */

    /** Attribute CPP1 Price represents the price of Energy, Gas, or Water
     *  delivered to the premises (i.e. delivered to the customer from the utility)
     *  while Critical Peak Pricing 'CPP1' is being applied.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_CPP1_PRICE = ESP_ZB_ZCL_ATTR_SET_WITH_ATTR_ID(ESP_ZB_ZCL_PRICE_SET_EXTENDED_PRICE_INFORMATION, 0x0FE),

    /** Attribute CPP2 Price represents the price of Energy, Gas, or Water
     *  delivered to the premises (i.e. delivered to the customer from the utility)
     *  while Critical Peak Pricing 'CPP2' is being applied.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_CPP2_PRICE,
} esp_zb_zcl_price_set_extended_price_information_t;


/** @brief Tariff Information (Delivered) Attributes Set */
typedef enum {
    /* reserved 0x0600 - 0x060F */

    /** The TariffLabel attribute provides a method for utilities to assign a
     *  label to an entire set of tariff information.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TARIFF_LABEL = ESP_ZB_ZCL_ATTR_SET_WITH_ATTR_ID(ESP_ZB_ZCL_PRICE_SET_TARIFF_INFORMATION, 0x10),

    /** An 8-bit integer which indicates the number of price tiers used while this
     *  tariff is active.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_NUMBER_OF_PRICE_TIERS_IN_USE,

    /** An 8-bit integer which indicates the total number of block thresholds used
     *  in the currently active tariff.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_NUMBER_OF_BLOCK_THRESHOLDS_IN_USE,

    /** An 8-bit enumeration indicating how the mixed TOU / Block charging is to
     *  be applied.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TIER_BLOCK_MODE,
    /* reserved 0x0614 */

    /** An 8-bit enumeration identifying the base unit of measure.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_UNIT_OF_MEASURE = ESP_ZB_ZCL_ATTR_SET_WITH_ATTR_ID(ESP_ZB_ZCL_PRICE_SET_TARIFF_INFORMATION, 0x15),

    /** An unsigned 16-bit integer containing identifying information concerning
     *  the local unit of currency used in the Price cluster.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_CURRENCY,

    /** An 8-bit BitMap used to determine where the decimal point is located for
     *  prices provided in the Standing Charge attribute and the Price Matrix command.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_PRICE_TRAILING_DIGIT,
    /* reserved 0x0618 */

    /** An 8-bit enumeration identifying the resolution period for Block Tariff.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_TARIFF_RESOLUTION_PERIOD = ESP_ZB_ZCL_ATTR_SET_WITH_ATTR_ID(ESP_ZB_ZCL_PRICE_SET_TARIFF_INFORMATION, 0x19),
    /* reserved 0x061A - 0x061F */

    /** Used to calculate the amount of carbon dioxide (CO2) produced from energy use.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_CO2 = ESP_ZB_ZCL_ATTR_SET_WITH_ATTR_ID(ESP_ZB_ZCL_PRICE_SET_TARIFF_INFORMATION, 0x20),

    /** This attribute is an 8-bit enumeration which defines the unit for the CO2 attribute.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_CO2_UNIT,

    /** An 8-bit Bit-Map used to determine where the decimal point is located in
     *  the CO2 attribute.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_CO2_TRAILING_DIGIT,
    /* reserved 0x628 - 0x06FF */
} esp_zb_zcl_price_set_tariff_information_t;

/** @brief Values for TariffResolutionPeriod attribute
 *  @see ESP_ZB_ZCL_ATTR_PRICE_SRV_TARIFF_RESOLUTION_PERIOD
 */
typedef enum {
    ESP_ZB_ZCL_ATTR_PRICE_TARIFF_RESOLUTION_PERIOD_NOT_DEFINED = 0x00,   /**< Not defined */
    ESP_ZB_ZCL_ATTR_PRICE_TARIFF_RESOLUTION_PERIOD_BLOCK_PERIOD,         /**< Block Period */
    ESP_ZB_ZCL_ATTR_PRICE_TARIFF_RESOLUTION_PERIOD_1DAY,                 /**< 1 Day */
} esp_zb_zcl_price_tariff_resolution_period_t;

/** @brief Values for CO2Unit attribute
 *  @see ESP_ZB_ZCL_ATTR_PRICE_SRV_CO2_UNIT
 */
typedef enum {
    ESP_ZB_ZCL_ATTR_PRICE_CO2_UNIT_KG_PER_KWH                  = 0x01,   /**< kg per kWh */
    ESP_ZB_ZCL_ATTR_PRICE_CO2_UNIT_KG_PER_GALLON_OF_GASOLINE   = 0x02,   /**< kg per Gallon of Gasoline */
    ESP_ZB_ZCL_ATTR_PRICE_CO2_UNIT_KG_PER_THERM_OF_NATURAL_GAS = 0x03,   /**< kg per Therm of Natural Gas */
} esp_zb_zcl_price_co2_unit_t;


/** @brief Billing Information (Delivered) Attributes Set */
typedef enum {
    /** The CurrentBillingPeriodStart attribute represents the start time of the
     *  current billing period.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_CURRENT_BILLING_PERIOD_START = ESP_ZB_ZCL_ATTR_SET_WITH_ATTR_ID(ESP_ZB_ZCL_PRICE_SET_BILLING_INFORMATION, 0x00),

    /** The CurrentBillingPeriodDuration attribute represents the current billing
     *  period duration in minutes.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_CURRENT_BILLING_PERIOD_DURATION,

    /** The LastBillingPeriodStart attribute represents the start time of the last
     *  billing period.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_LAST_BILLING_PERIOD_START,

    /** The LastBillingPeriodDuration attribute is the duration of the last
     *  billing period in minutes (start to end of last billing period).
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_LAST_BILLING_PERIOD_DURATION,

    /** The LastBillingPeriodConsolidatedBill attribute is an amount for the cost
     *  of the energy supplied from the date of the LastBillingPeriodStart
     *  attribute and until the duration of the LastBillingPeriodDuration attribute
     *  expires, measured in base unit of Currency with the decimal point located
     *  as indicated by the Trailing Digits attribute.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_LAST_BILLING_PERIOD_CONSOLIDATED_BILL,
    /* reserved 0x0705 - 0x07FF */
} esp_zb_zcl_price_set_billing_information_t;


/** @brief Credit Payment Attributes Set */
typedef enum {
    /** The CreditPaymentDueDate attribute indicates the date and time when the
     *  next credit payment is due to be paid by the consumer to the supplier.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_CREDIT_PAYMENT_DUE_DATE = ESP_ZB_ZCL_ATTR_SET_WITH_ATTR_ID(ESP_ZB_ZCL_PRICE_SET_CREDIT_PAYMENT, 0x00),

    /** The CreditPaymentStatus attribute indicates the current status of the last payment.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_CREDIT_PAYMENT_STATUS,

    /** This is the total of the consolidated bill amounts accumulated since the
     *  last payment.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_CREDIT_PAYMENT_OVER_DUE_AMOUNT,
    /* reserved 0x0803 - 0x0809 */

    /** The PaymentDiscount attribute indicates the discount that the energy
     *  supplier has applied to the consolidated bill.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_CREDIT_PAYMENT = ESP_ZB_ZCL_ATTR_SET_WITH_ATTR_ID(ESP_ZB_ZCL_PRICE_SET_CREDIT_PAYMENT, 0x0A),

    /** The PaymentDiscountPeriod attribute indicates the period for which this
     *  discount shall be applied for.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_CREDIT_PAYMENT_PERIOD,
    /* reserved 0x080C - 0x080F */

    /** The CreditPayment attributes indicate the amount paid by the consumer to
     *  the energy supplier.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_CREDIT_PAYMENT_1 = ESP_ZB_ZCL_ATTR_SET_WITH_ATTR_ID(ESP_ZB_ZCL_PRICE_SET_CREDIT_PAYMENT, 0x10),

    /** The CreditPaymentDate attributes indicate the last time the consumer made
     *  a payment to the energy supplier.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_CREDIT_PAYMENT_DATE_1,

    /** The CreditPaymentRef attributes indicate the reference number given to the
     *  payment by the energy supplier.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_CREDIT_PAYMENT_REF_1,
    /* reserved 0x0813 - 0x081F */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_CREDIT_PAYMENT_2 = ESP_ZB_ZCL_ATTR_SET_WITH_ATTR_ID(ESP_ZB_ZCL_PRICE_SET_CREDIT_PAYMENT, 0x20),
    ESP_ZB_ZCL_ATTR_PRICE_SRV_CREDIT_PAYMENT_DATE_2,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_CREDIT_PAYMENT_REF_2,
    /* reserved 0x0823 - 0x082F */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_CREDIT_PAYMENT_3 = ESP_ZB_ZCL_ATTR_SET_WITH_ATTR_ID(ESP_ZB_ZCL_PRICE_SET_CREDIT_PAYMENT, 0x30),
    ESP_ZB_ZCL_ATTR_PRICE_SRV_CREDIT_PAYMENT_DATE_3,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_CREDIT_PAYMENT_REF_3,
    /* reserved 0x0833 - 0x083F */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_CREDIT_PAYMENT_4 = ESP_ZB_ZCL_ATTR_SET_WITH_ATTR_ID(ESP_ZB_ZCL_PRICE_SET_CREDIT_PAYMENT, 0x40),
    ESP_ZB_ZCL_ATTR_PRICE_SRV_CREDIT_PAYMENT_DATE_4,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_CREDIT_PAYMENT_REF_4,
    /* reserved 0x0843 - 0x084F */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_CREDIT_PAYMENT_5 = ESP_ZB_ZCL_ATTR_SET_WITH_ATTR_ID(ESP_ZB_ZCL_PRICE_SET_CREDIT_PAYMENT, 0x50),
    ESP_ZB_ZCL_ATTR_PRICE_SRV_CREDIT_PAYMENT_DATE_5,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_CREDIT_PAYMENT_REF_5,
    /* reserved 0x0853 - 0x08FF */
} esp_zb_zcl_price_set_credit_payment_t;

/** @brief Values for CreditPaymentStatus attribute
 *  @see ESP_ZB_ZCL_ATTR_PRICE_SRV_CREDIT_PAYMENT_STATUS
 */
typedef enum {
    ESP_ZB_ZCL_PRICE_CREDIT_PAYMENT_PENDING            = 0x00,    /**< Pending */
    ESP_ZB_ZCL_PRICE_CREDIT_PAYMENT_RECEIVED           = 0x01,    /**< Received/Paid */
    ESP_ZB_ZCL_PRICE_CREDIT_PAYMENT_PAID               = ESP_ZB_ZCL_PRICE_CREDIT_PAYMENT_RECEIVED,    /**< Payment paid */
    ESP_ZB_ZCL_PRICE_CREDIT_PAYMENT_OVERDUE            = 0x02,    /**< Overdue */
    ESP_ZB_ZCL_PRICE_CREDIT_PAYMENT_2PAYMENTS_OVERDUE  = 0x03,    /**< 2 Payments overdue */
    ESP_ZB_ZCL_PRICE_CREDIT_PAYMENT_3PAYMENTS_OVERDUE  = 0x04,    /**< 3 Payments overdue */
    /* reserved 0x05 - 0xFF */
} esp_zb_zcl_price_credit_payment_status_t;


/** @brief Received Tier Label Attributes Set */
typedef enum {
    /** The ReceivedTierNPriceLabel attributes provide a method for utilities to
     *  assign a label to Received Price Tiers.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER1_PRICE_LABEL = ESP_ZB_ZCL_ATTR_SET_WITH_ATTR_ID(ESP_ZB_ZCL_PRICE_SET_RECEIVED_TIER_LABEL, 0x00),
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER2_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER3_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER4_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER5_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER6_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER7_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER8_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER9_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER10_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER11_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER12_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER13_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER14_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER15_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER16_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER17_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER18_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER19_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER20_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER21_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER22_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER23_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER24_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER25_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER26_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER27_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER28_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER29_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER30_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER31_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER32_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER33_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER34_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER35_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER36_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER37_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER38_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER39_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER40_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER41_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER42_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER43_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER44_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER45_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER46_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER47_PRICE_LABEL,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER48_PRICE_LABEL,
    /* reserved 0x8030 - 0x80FF */
} esp_zb_zcl_price_set_received_tier_label_t;


/** @brief Received Block Threshold Attributes Set */
typedef enum {
    /** The format of these attributes is the same as for the 'Delivered' Block Thresholds
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_BLOCK1_THRESHOLD = ESP_ZB_ZCL_ATTR_SET_WITH_ATTR_ID(ESP_ZB_ZCL_PRICE_SET_RECEIVED_BLOCK_THRESHOLD, 0x00),
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_BLOCK2_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_BLOCK3_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_BLOCK4_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_BLOCK5_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_BLOCK6_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_BLOCK7_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_BLOCK8_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_BLOCK9_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_BLOCK10_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_BLOCK11_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_BLOCK12_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_BLOCK13_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_BLOCK14_THRESHOLD,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_BLOCK15_THRESHOLD,
    /* reserved 0x810F - 0x81FF */
} esp_zb_zcl_price_set_received_block_threshold_t;


/** @brief Received Block Period Attributes Set */
typedef enum {
    /** The format of this attribute is the same as for the 'Delivered'
     *  StartOfBlockPeriod attribute.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_START_OF_BLOCK_PERIOD = ESP_ZB_ZCL_ATTR_SET_WITH_ATTR_ID(ESP_ZB_ZCL_PRICE_SET_RECEIVED_BLOCK_PERIOD, 0x00),

    /** The format of this attribute is the same as for the 'Delivered'
     *  BlockPeriodDuration attribute.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_BLOCK_PERIOD_DURATION,

    /** The format of this attribute is the same as for the 'Delivered'
     *  ThresholdMultiplier attribute.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_THRESHOLD_MULTIPLIER,

    /** The format of this attribute is the same as for the 'Delivered'
     *  ThresholdDivisor attribute
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_THRESHOLD_DIVISOR,
    /* reserved 0x8204 - 0x82FF */
} esp_zb_zcl_price_set_received_block_period_t;


/** @brief Received Block Price Information Attributes Set */
typedef enum {
    /** The format and use of these attributes is the same as for the
     *  'Delivered' TierNBlockNPrice attributes.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_NO_TIER_BLOCK1_PRICE = ESP_ZB_ZCL_ATTR_SET_WITH_ATTR_ID(ESP_ZB_ZCL_PRICE_SET_RECEIVED_BLOCK_PRICE_INFORMATION, 0x00),
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_NO_TIER_BLOCK2_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_NO_TIER_BLOCK3_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_NO_TIER_BLOCK4_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_NO_TIER_BLOCK5_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_NO_TIER_BLOCK6_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_NO_TIER_BLOCK7_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_NO_TIER_BLOCK8_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_NO_TIER_BLOCK9_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_NO_TIER_BLOCK10_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_NO_TIER_BLOCK11_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_NO_TIER_BLOCK12_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_NO_TIER_BLOCK13_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_NO_TIER_BLOCK14_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_NO_TIER_BLOCK15_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_NO_TIER_BLOCK16_PRICE,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER1_BLOCK1_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER1_BLOCK2_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER1_BLOCK3_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER1_BLOCK4_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER1_BLOCK5_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER1_BLOCK6_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER1_BLOCK7_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER1_BLOCK8_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER1_BLOCK9_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER1_BLOCK10_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER1_BLOCK11_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER1_BLOCK12_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER1_BLOCK13_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER1_BLOCK14_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER1_BLOCK15_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER1_BLOCK16_PRICE,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER2_BLOCK1_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER2_BLOCK2_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER2_BLOCK3_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER2_BLOCK4_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER2_BLOCK5_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER2_BLOCK6_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER2_BLOCK7_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER2_BLOCK8_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER2_BLOCK9_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER2_BLOCK10_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER2_BLOCK11_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER2_BLOCK12_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER2_BLOCK13_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER2_BLOCK14_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER2_BLOCK15_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER2_BLOCK16_PRICE,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER3_BLOCK1_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER3_BLOCK2_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER3_BLOCK3_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER3_BLOCK4_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER3_BLOCK5_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER3_BLOCK6_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER3_BLOCK7_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER3_BLOCK8_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER3_BLOCK9_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER3_BLOCK10_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER3_BLOCK11_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER3_BLOCK12_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER3_BLOCK13_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER3_BLOCK14_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER3_BLOCK15_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER3_BLOCK16_PRICE,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER4_BLOCK1_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER4_BLOCK2_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER4_BLOCK3_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER4_BLOCK4_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER4_BLOCK5_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER4_BLOCK6_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER4_BLOCK7_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER4_BLOCK8_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER4_BLOCK9_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER4_BLOCK10_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER4_BLOCK11_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER4_BLOCK12_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER4_BLOCK13_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER4_BLOCK14_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER4_BLOCK15_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER4_BLOCK16_PRICE,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER5_BLOCK1_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER5_BLOCK2_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER5_BLOCK3_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER5_BLOCK4_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER5_BLOCK5_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER5_BLOCK6_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER5_BLOCK7_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER5_BLOCK8_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER5_BLOCK9_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER5_BLOCK10_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER5_BLOCK11_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER5_BLOCK12_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER5_BLOCK13_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER5_BLOCK14_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER5_BLOCK15_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER5_BLOCK16_PRICE,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER6_BLOCK1_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER6_BLOCK2_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER6_BLOCK3_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER6_BLOCK4_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER6_BLOCK5_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER6_BLOCK6_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER6_BLOCK7_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER6_BLOCK8_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER6_BLOCK9_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER6_BLOCK10_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER6_BLOCK11_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER6_BLOCK12_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER6_BLOCK13_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER6_BLOCK14_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER6_BLOCK15_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER6_BLOCK16_PRICE,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER7_BLOCK1_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER7_BLOCK2_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER7_BLOCK3_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER7_BLOCK4_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER7_BLOCK5_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER7_BLOCK6_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER7_BLOCK7_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER7_BLOCK8_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER7_BLOCK9_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER7_BLOCK10_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER7_BLOCK11_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER7_BLOCK12_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER7_BLOCK13_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER7_BLOCK14_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER7_BLOCK15_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER7_BLOCK16_PRICE,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER8_BLOCK1_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER8_BLOCK2_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER8_BLOCK3_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER8_BLOCK4_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER8_BLOCK5_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER8_BLOCK6_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER8_BLOCK7_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER8_BLOCK8_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER8_BLOCK9_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER8_BLOCK10_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER8_BLOCK11_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER8_BLOCK12_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER8_BLOCK13_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER8_BLOCK14_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER8_BLOCK15_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER8_BLOCK16_PRICE,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER9_BLOCK1_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER9_BLOCK2_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER9_BLOCK3_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER9_BLOCK4_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER9_BLOCK5_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER9_BLOCK6_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER9_BLOCK7_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER9_BLOCK8_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER9_BLOCK9_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER9_BLOCK10_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER9_BLOCK11_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER9_BLOCK12_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER9_BLOCK13_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER9_BLOCK14_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER9_BLOCK15_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER9_BLOCK16_PRICE,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER10_BLOCK1_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER10_BLOCK2_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER10_BLOCK3_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER10_BLOCK4_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER10_BLOCK5_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER10_BLOCK6_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER10_BLOCK7_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER10_BLOCK8_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER10_BLOCK9_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER10_BLOCK10_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER10_BLOCK11_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER10_BLOCK12_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER10_BLOCK13_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER10_BLOCK14_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER10_BLOCK15_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER10_BLOCK16_PRICE,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER11_BLOCK1_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER11_BLOCK2_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER11_BLOCK3_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER11_BLOCK4_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER11_BLOCK5_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER11_BLOCK6_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER11_BLOCK7_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER11_BLOCK8_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER11_BLOCK9_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER11_BLOCK10_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER11_BLOCK11_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER11_BLOCK12_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER11_BLOCK13_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER11_BLOCK14_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER11_BLOCK15_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER11_BLOCK16_PRICE,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER12_BLOCK1_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER12_BLOCK2_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER12_BLOCK3_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER12_BLOCK4_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER12_BLOCK5_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER12_BLOCK6_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER12_BLOCK7_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER12_BLOCK8_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER12_BLOCK9_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER12_BLOCK10_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER12_BLOCK11_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER12_BLOCK12_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER12_BLOCK13_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER12_BLOCK14_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER12_BLOCK15_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER12_BLOCK16_PRICE,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER13_BLOCK1_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER13_BLOCK2_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER13_BLOCK3_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER13_BLOCK4_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER13_BLOCK5_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER13_BLOCK6_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER13_BLOCK7_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER13_BLOCK8_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER13_BLOCK9_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER13_BLOCK10_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER13_BLOCK11_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER13_BLOCK12_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER13_BLOCK13_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER13_BLOCK14_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER13_BLOCK15_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER13_BLOCK16_PRICE,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER14_BLOCK1_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER14_BLOCK2_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER14_BLOCK3_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER14_BLOCK4_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER14_BLOCK5_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER14_BLOCK6_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER14_BLOCK7_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER14_BLOCK8_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER14_BLOCK9_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER14_BLOCK10_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER14_BLOCK11_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER14_BLOCK12_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER14_BLOCK13_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER14_BLOCK14_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER14_BLOCK15_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER14_BLOCK16_PRICE,

    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER15_BLOCK1_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER15_BLOCK2_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER15_BLOCK3_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER15_BLOCK4_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER15_BLOCK5_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER15_BLOCK6_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER15_BLOCK7_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER15_BLOCK8_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER15_BLOCK9_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER15_BLOCK10_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER15_BLOCK11_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER15_BLOCK12_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER15_BLOCK13_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER15_BLOCK14_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER15_BLOCK15_PRICE,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RX_TIER15_BLOCK16_PRICE,
} esp_zb_zcl_price_set_received_block_price_information_t;


/** @brief Received Extended Price Information Attributes Set */
typedef enum {
    /* reserved 0x8500 - 0x850E */

    /** The format and use of these attributes is the same as for the
     *  'Delivered' PriceTierN attributes.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_PRICE_TIER16 = ESP_ZB_ZCL_ATTR_SET_WITH_ATTR_ID(ESP_ZB_ZCL_PRICE_SET_RECEIVED_EXTENDED_PRICE_INFORMATION, 0x0F),
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_PRICE_TIER17,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_PRICE_TIER18,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_PRICE_TIER19,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_PRICE_TIER20,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_PRICE_TIER21,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_PRICE_TIER22,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_PRICE_TIER23,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_PRICE_TIER24,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_PRICE_TIER25,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_PRICE_TIER26,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_PRICE_TIER27,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_PRICE_TIER28,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_PRICE_TIER29,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_PRICE_TIER30,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_PRICE_TIER31,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_PRICE_TIER32,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_PRICE_TIER33,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_PRICE_TIER34,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_PRICE_TIER35,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_PRICE_TIER36,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_PRICE_TIER37,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_PRICE_TIER38,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_PRICE_TIER39,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_PRICE_TIER40,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_PRICE_TIER41,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_PRICE_TIER42,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_PRICE_TIER43,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_PRICE_TIER44,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_PRICE_TIER45,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_PRICE_TIER46,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_PRICE_TIER47,
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_PRICE_TIER48,
    /* reserved 0x8530 - 0x85FF */
} esp_zb_zcl_price_set_received_extended_price_information_t;


/** @brief Received Tariff Information Attributes Set */
typedef enum {
    /* reserved 0x8600 - 0x860F */

    /** The format and use of this attribute is the same as for the
     *  'Delivered' TariffLabel attribute.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TARIFF_LABEL = ESP_ZB_ZCL_ATTR_SET_WITH_ATTR_ID(ESP_ZB_ZCL_PRICE_SET_RECEIVED_TARIFF_INFORMATION, 0x10),

    /** The format and use of this attribute is the same as for the
     *  'Delivered' NumberOfPriceTiersInUse attribute.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_NUMBER_OF_PRICE_TIERS_IN_USE,

    /** The format and use of this attribute is the same as for the
     *  'Delivered' NumberOfBlockThresholdsInUse attribute.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_NUMBER_OF_BLOCK_THRESHOLDS_IN_USE,

    /** The format and use of this attribute is the same as for the
     *  'Delivered' TierBlockMode attribute.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TIER_BLOCK_MODE,
    /* reserved 0x8614 */

    /** An 8-bit enumeration identifying the resolution period for Block Tariff.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_TARIFF_RESOLUTION_PERIOD = ESP_ZB_ZCL_ATTR_SET_WITH_ATTR_ID(ESP_ZB_ZCL_PRICE_SET_RECEIVED_TARIFF_INFORMATION, 0x15),
    /* reserved 0x8616 - 0x8624 */

    /** The format and use of this attribute is the same as for the
     *  'Delivered' CO2 attribute.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_CO2 = ESP_ZB_ZCL_ATTR_SET_WITH_ATTR_ID(ESP_ZB_ZCL_PRICE_SET_RECEIVED_TARIFF_INFORMATION, 0x25),

    /** The format and use of this attribute is the same as for the
     *  'Delivered' CO2Unit attribute.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_CO2_UNIT,

    /** The format and use of this attribute is the same as for the
     *  'Delivered' CO2TrailingDigit attribute.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_CO2_TRAILING_DIGIT,
    /* reserved 0x8528 - 0x85FF */
} esp_zb_zcl_price_set_received_tariff_information_t;


/** @brief Received Billing Information Attributes Set */
typedef enum {
    /** The format and use of this attribute is the same as for the
     *  'Delivered' CurrentBillingPeriodStart attribute.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_CURRENT_BILLING_PERIOD_START = ESP_ZB_ZCL_ATTR_SET_WITH_ATTR_ID(ESP_ZB_ZCL_PRICE_SET_RECEIVED_BILLING_INFORMATION, 0x00),

    /** The format and use of this attribute is the same as for the
     *  'Delivered' CurrentBillingPeriodDuration attribute.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_CURRENT_BILLING_PERIOD_DURATION,

    /** The format and use of this attribute is the same as for the
     *  'Delivered' LastBillingPeriodStart attribute.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_LAST_BILLING_PERIOD_START,

    /** The format and use of this attribute is the same as for the
     *  'Delivered' LastBillingPeriodDuration attribute.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_LAST_BILLING_PERIOD_DURATION,

    /** The format and use of this attribute is the same as for the
     *  'Delivered' LastBillingPeriodConsolidatedBill attribute.
     */
    ESP_ZB_ZCL_ATTR_PRICE_SRV_RECEIVED_LAST_BILLING_PERIOD_CONSOLIDATED_BILL,
    /* reserved 0x8705 - 0x87FF */
} esp_zb_zcl_price_set_received_billing_information_t;


/** @brief Default value for Price cluster revision global attribute */
#define ESP_ZB_ZCL_PRICE_CLUSTER_REVISION_DEFAULT ((uint16_t)0x0002u)

/** @brief Default value for TierN_PriceLabel attribute */
#define ESP_ZB_ZCL_PRICE_SRV_TIERN_PRICE_LABEL_DEFAULT_VALUE(_tierN) \
    {sizeof("Tier0"), 'T', 'i', 'e', 'r', '0' + (_tierN), 0x00}

/** @brief Default value for TierNM_PriceLabel attribute */
#define ESP_ZB_ZCL_PRICE_SRV_TIERNM_PRICE_LABEL_DEFAULT_VALUE(_tierN, _tierM) \
    {sizeof("Tier00"), 'T', 'i', 'e', 'r', '0' + (_tierN), '0' + (_tierM), 0x00}

/** @brief Default value for ConversionFactor attribute */
#define ESP_ZB_ZCL_PRICE_SRV_CONVERSION_FACTOR_DEFAULT_VALUE ((uint32_t)0x10000000)

/** @brief Default value for ConversionFactorTrailingDigit attribute */
#define ESP_ZB_ZCL_PRICE_SRV_CONVERSION_FACTOR_TRAILING_DIGIT_DEFAULT_VALUE ((uint8_t)0x70)

/** @brief Default value for CalorificValue attribute */
#define ESP_ZB_ZCL_PRICE_SRV_CALORIFIC_VALUE_DEFAULT_VALUE ((uint32_t)0x2625A00)

/** @brief Default value for CalorificValueUnit attribute */
#define ESP_ZB_ZCL_PRICE_SRV_CALORIFIC_VALUE_UNIT_DEFAULT_VALUE ((uint8_t)0x1)

/** @brief Default value for CalorificValueTrailingDigit attribute */
#define ESP_ZB_ZCL_PRICE_SRV_CALORIFIC_VALUE_TRAILING_DIGIT_DEFAULT_VALUE ((uint8_t)0x60)


/** @brief Price cluster client command identifiers. */
typedef enum {
    ESP_ZB_ZCL_PRICE_CLI_CMD_GET_CURRENT_PRICE            = 0x00,   /**< GetCurrentPrice command initiates a PublishPrice command for the current time. */
    ESP_ZB_ZCL_PRICE_CLI_CMD_GET_SCHEDULED_PRICES         = 0x01,   /**< GetScheduledPrices command initiates a PublishPrice command for available price events. */
    ESP_ZB_ZCL_PRICE_CLI_CMD_PRICE_ACK                    = 0x02,   /**< PriceAcknowledgement command provides the ability to acknowledge a previously sent PublishPrice command. */
    ESP_ZB_ZCL_PRICE_CLI_CMD_GET_BLOCK_PERIOD             = 0x03,   /**< GetBlockPeriod(s) command initiates a PublishBlockPeriod command for the currently scheduled block periods. */
    ESP_ZB_ZCL_PRICE_CLI_CMD_GET_CONVERSION_FACTOR        = 0x04,   /**< GetConversionFactor command initiates a PublishConversionFactor command(s) for scheduled conversion factor updates. */
    ESP_ZB_ZCL_PRICE_CLI_CMD_GET_CALORIFIC_VALUE          = 0x05,   /**< GetCalorificValue command initiates a PublishCalorificValue command(s) for scheduled calorific value updates. */
    ESP_ZB_ZCL_PRICE_CLI_CMD_GET_TARIFF_INFORMATION       = 0x06,   /**< GetTariffInformation command initiates PublishTariffInformation command(s) for scheduled tariff updates. */
    ESP_ZB_ZCL_PRICE_CLI_CMD_GET_PRICE_MATRIX             = 0x07,   /**< GetPriceMatrix command initiates a PublishPriceMatrix command for the scheduled Price Matrix updates. */
    ESP_ZB_ZCL_PRICE_CLI_CMD_GET_BLOCK_THRESHOLDS         = 0x08,   /**< GetBlockThresholds command initiates a PublishBlockThreshold command for the scheduled Block Threshold updates. */
    ESP_ZB_ZCL_PRICE_CLI_CMD_GET_CO2_VALUE                = 0x09,   /**< GetCO2Value command initiates PublishCO2Value command(s) for scheduled CO2 conversion factor updates. */
    ESP_ZB_ZCL_PRICE_CLI_CMD_GET_TIER_LABELS              = 0x0A,   /**< GetTierLabels command allows a client to retrieve the tier labels associated with a given tariff and initiates a PublishTierLabels command from the server. */
    ESP_ZB_ZCL_PRICE_CLI_CMD_GET_BILLING_PERIOD           = 0x0B,   /**< GetBillingPeriod command initiates one or more PublishBillingPeriod commands for currently scheduled billing periods. */
    ESP_ZB_ZCL_PRICE_CLI_CMD_GET_CONSOLIDATED_BILL        = 0x0C,   /**< GetConsolidatedBill command initiates one or more PublishConsolidatedBill commands with the requested billing information. */
    ESP_ZB_ZCL_PRICE_CLI_CMD_CPP_EVENT_RESPONSE           = 0x0D,   /**< The CPPEventResponse command is sent from a Client (IHD) to the ESI to notify it of a Critical Peak Pricing event authorization. */
    ESP_ZB_ZCL_PRICE_CLI_CMD_GET_CREDIT_PAYMENT           = 0x0E,   /**< GetCreditPayment command initiates PublishCreditPayment commands for the requested credit payment information. */
    ESP_ZB_ZCL_PRICE_CLI_CMD_GET_CURRENCY_CONVERSION      = 0x0F,   /**< GetCurrencyConversion command initiates a PublishCurrencyConversion command for the currency conversion factor updates. */
    ESP_ZB_ZCL_PRICE_CLI_CMD_GET_TARIFF_CANCELLATION      = 0x10,   /**< GetTariffCancellation command initiates the return of the last CancelTariff command held on the associated server. */
} esp_zb_zcl_price_cli_cmd_t;

/** @brief Price cluster server command identifiers */
typedef enum {
    ESP_ZB_ZCL_PRICE_SRV_CMD_PUBLISH_PRICE                = 0x00,   /**< PublishPrice command. */
    ESP_ZB_ZCL_PRICE_SRV_CMD_PUBLISH_BLOCK_PERIOD         = 0x01,   /**< PublishBlockPeriod command. */
    ESP_ZB_ZCL_PRICE_SRV_CMD_PUBLISH_CONVERSION_FACTOR    = 0x02,   /**< PublishConversionFactor command. */
    ESP_ZB_ZCL_PRICE_SRV_CMD_PUBLISH_CALORIFIC_VALUE      = 0x03,   /**< PublishCalorificValue command. */
    ESP_ZB_ZCL_PRICE_SRV_CMD_PUBLISH_TARIFF_INFORMATION   = 0x04,   /**< PublishTariffInformation command. */
    ESP_ZB_ZCL_PRICE_SRV_CMD_PUBLISH_PRICE_MATRIX         = 0x05,   /**< PublishPriceMatrix command. */
    ESP_ZB_ZCL_PRICE_SRV_CMD_PUBLISH_BLOCK_THRESHOLDS     = 0x06,   /**< PublishBlockThresholds command. */
    ESP_ZB_ZCL_PRICE_SRV_CMD_PUBLISH_CO2_VALUE            = 0x07,   /**< PublishCO2Value command. */
    ESP_ZB_ZCL_PRICE_SRV_CMD_PUBLISH_TIER_LABELS          = 0x08,   /**< PublishTierLabels command. */
    ESP_ZB_ZCL_PRICE_SRV_CMD_PUBLISH_BILLING_PERIOD       = 0x09,   /**< PublishBillingPeriod command. */
    ESP_ZB_ZCL_PRICE_SRV_CMD_PUBLISH_CONSOLIDATED_BILL    = 0x0A,   /**< PublishConsolidatedBill command. */
    ESP_ZB_ZCL_PRICE_SRV_CMD_PUBLISH_CPP_EVENT            = 0x0B,   /** PublishCPPEvent command. */
    ESP_ZB_ZCL_PRICE_SRV_CMD_PUBLISH_CREDIT_PAYMENT       = 0x0C,   /**< PublishCreditPayment command. */
    ESP_ZB_ZCL_PRICE_SRV_CMD_PUBLISH_CURRENCY_CONVERSION  = 0x0D,   /**< PublishCurrencyConversion command. */
    ESP_ZB_ZCL_PRICE_SRV_CMD_CANCEL_TARIFF                = 0x0E,   /**< CancelTariff command. */
} esp_zb_zcl_price_srv_cmd_t;

#ifdef __cplusplus
}
#endif
