/*
 * Decompiled with CFR 0.152.
 */
package org.skills.data.managers.backup;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Enumeration;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipOutputStream;
import org.skills.main.locale.MessageHandler;
import org.skills.utils.StringUtils;

public abstract class BackupManager {
    protected static final String TIME_PATTERN = "yyyy-MM-dd";
    protected static boolean useMultiBackups = true;
    protected final Path backups;
    protected final Path toBackup;

    public BackupManager(File file, File file2) {
        Objects.requireNonNull(file, "Backups directory cannot be null");
        Objects.requireNonNull(file2, "Cannot backup null directory");
        this.backups = file.toPath();
        this.validateDir();
        this.toBackup = file2.toPath();
    }

    private void validateDir() {
        try {
            Files.createDirectories(this.backups, new FileAttribute[0]);
        }
        catch (IOException iOException) {
            MessageHandler.sendConsolePluginMessage("&4Failed to create backups directory.");
            iOException.printStackTrace();
        }
    }

    public String getDate() {
        return LocalDateTime.now().format(DateTimeFormatter.ofPattern(TIME_PATTERN));
    }

    public CompletableFuture<Void> unzipFiles(Path path, Path path2) {
        Objects.requireNonNull(path, "Cannot unzip null directory.");
        Objects.requireNonNull(path2, "Cannot unzip to null directory");
        if (!path.toString().toLowerCase().endsWith(".zip")) {
            throw new IllegalArgumentException("ZIP path must refer to a ZIP file");
        }
        if (Files.exists(path2, new LinkOption[0]) && !Files.isDirectory(path2, new LinkOption[0])) {
            throw new IllegalArgumentException("Cannot unzip to a non-directory");
        }
        return CompletableFuture.runAsync(() -> {
            try {
                Files.createDirectories(path2, new FileAttribute[0]);
            }
            catch (IOException iOException) {
                iOException.printStackTrace();
            }
            try (ZipFile zipFile = new ZipFile(path.toString());){
                Enumeration<? extends ZipEntry> enumeration = zipFile.entries();
                while (enumeration.hasMoreElements()) {
                    ZipEntry zipEntry = enumeration.nextElement();
                    InputStream inputStream = zipFile.getInputStream(zipEntry);
                    try {
                        Path path3 = path2.resolve(zipEntry.getName());
                        Files.createDirectories(path3, new FileAttribute[0]);
                        Files.copy(inputStream, path3, StandardCopyOption.REPLACE_EXISTING);
                    }
                    finally {
                        if (inputStream == null) continue;
                        inputStream.close();
                    }
                }
            }
            catch (IOException iOException) {
                iOException.printStackTrace();
            }
        });
    }

    public void takeBackup() {
        if (!useMultiBackups && this.hasBackupToday()) {
            return;
        }
        this.deleteOldBackups(30, TimeUnit.DAYS);
        this.zipFiles();
    }

    public CompletableFuture<Integer> zipFiles() {
        this.validateDir();
        final AtomicInteger atomicInteger = new AtomicInteger();
        Path path = this.getZip();
        try {
            Files.createFile(path, new FileAttribute[0]);
        }
        catch (IOException iOException) {
            MessageHandler.sendConsolePluginMessage("&4Error while attempting to create ZIP file.");
            iOException.printStackTrace();
        }
        return CompletableFuture.supplyAsync(() -> {
            try (final ZipOutputStream zipOutputStream = new ZipOutputStream(new BufferedOutputStream(Files.newOutputStream(path, new OpenOption[0])));){
                zipOutputStream.setLevel(9);
                zipOutputStream.setComment("A backup file for Skills minecraft plugin data.\nThese backups contain language file, config.yml and players data\ndepending on the options specified in the config.\n\nNote that you have to stop the server before restoring one of these backups.\nBackup taken at: " + StringUtils.getFullTime());
                Files.walkFileTree(this.toBackup, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                    @Override
                    public FileVisitResult preVisitDirectory(Path path, BasicFileAttributes basicFileAttributes) {
                        return BackupManager.this.isWhitelistedDirectory(path) || path == BackupManager.this.toBackup ? FileVisitResult.CONTINUE : FileVisitResult.SKIP_SUBTREE;
                    }

                    @Override
                    public FileVisitResult visitFile(Path path, BasicFileAttributes basicFileAttributes) {
                        if (!BackupManager.this.isWhitelistedFile(path)) {
                            return FileVisitResult.SKIP_SUBTREE;
                        }
                        ZipEntry zipEntry = new ZipEntry(BackupManager.this.toBackup.relativize(path).toString());
                        try {
                            zipOutputStream.putNextEntry(zipEntry);
                            Files.copy(path, zipOutputStream);
                            zipOutputStream.closeEntry();
                            atomicInteger.getAndIncrement();
                        }
                        catch (IOException iOException) {
                            MessageHandler.sendConsolePluginMessage("&4Error while attempting to backup a file&8: &e" + path.getFileName());
                            iOException.printStackTrace();
                        }
                        return FileVisitResult.CONTINUE;
                    }
                });
            }
            catch (IOException iOException) {
                MessageHandler.sendConsolePluginMessage("&4Error while attempting to take a backup...");
                iOException.printStackTrace();
            }
            return atomicInteger.get();
        });
    }

    public Path getMultiZipName() {
        int n = 1;
        Path path = this.getZipPath();
        Path path2 = path.getParent();
        String string = path.getFileName().toString();
        string = string.substring(0, string.lastIndexOf(46));
        String string2 = ".zip";
        while (Files.exists(path, new LinkOption[0])) {
            path = path2.resolve(string + " (" + n++ + ")" + string2);
        }
        return path;
    }

    public abstract boolean isWhitelistedDirectory(Path var1);

    public abstract boolean isWhitelistedFile(Path var1);

    public boolean shouldBeDeleted(Path path, int n, TimeUnit timeUnit) {
        long l;
        try {
            l = Files.getLastModifiedTime(path, new LinkOption[0]).to(TimeUnit.MILLISECONDS);
        }
        catch (IOException iOException) {
            iOException.printStackTrace();
            return false;
        }
        long l2 = System.currentTimeMillis() - l;
        return l2 >= timeUnit.toMillis(n);
    }

    public CompletableFuture<Void> deleteOldBackups(int n, TimeUnit timeUnit) {
        return CompletableFuture.runAsync(() -> {
            try {
                Files.walk(this.backups, new FileVisitOption[0]).filter(path -> Files.isRegularFile(path, new LinkOption[0])).forEach(path -> {
                    if (this.shouldBeDeleted((Path)path, n, timeUnit)) {
                        try {
                            MessageHandler.sendConsolePluginMessage("&2Deleting old backup... &6" + path.getFileName());
                            Files.delete(path);
                        }
                        catch (IOException iOException) {
                            iOException.printStackTrace();
                        }
                    }
                });
            }
            catch (IOException iOException) {
                iOException.printStackTrace();
            }
        });
    }

    public Path getZipPath() {
        return this.backups.resolve(this.getDate() + ".zip");
    }

    public Path getZip() {
        return useMultiBackups ? this.getMultiZipName() : this.getZipPath();
    }

    public boolean hasBackupToday() {
        return Files.exists(this.getZip(), new LinkOption[0]);
    }
}

